<?php

namespace App\Http\Controllers\Profile;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\ValidationException;

class ProfileController extends Controller
{
    public function getProfile()
    {
        $user = Auth::user();

        return response()->json([
            'message' => 'Profil berhasil diambil.',
            'user' => [
                'id'          => $user->id,
                'name'        => $user->name,
                'username'    => $user->username,
                'first_name'  => $user->first_name,
                'last_name'   => $user->last_name,
                'email'       => $user->email,
                'whatsapp'    => $user->whatsapp,
                'avatar'      => $user->avatar,
                'google_id'   => $user->google_id,
                'created_at'  => $user->created_at,
                'updated_at'  => $user->updated_at,
            ],
        ]);
    }

    public function changeEmail(Request $request)
    {
        $request->validate([
            'current_email' => ['required','email'],
            'new_email'     => ['required','email','different:current_email','unique:users,email'],
        ]);

        $user = Auth::user();

        // Pastikan email lama cocok
        if ($user->email !== $request->current_email) {
            throw ValidationException::withMessages([
                'current_email' => 'Email lama tidak sesuai dengan akun Anda.',
            ]);
        }

        // Update email
        $user->email = $request->new_email;
        $user->email_verified_at = null; // agar user perlu verifikasi ulang kalau pakai sistem verifikasi
        $user->save();

        return response()->json([
            'message' => 'Email berhasil diubah.',
            'user'    => $user,
        ]);
    }

    public function changePassword(Request $request)
    {
        $request->validate([
            'current_password' => ['required'],
            'new_password'     => ['required','string','min:8','confirmed'], // butuh new_password_confirmation
        ]);

        $user = Auth::user();

        // Pastikan password lama benar
        if (!Hash::check($request->current_password, $user->password)) {
            throw ValidationException::withMessages([
                'current_password' => 'Password lama tidak sesuai.',
            ]);
        }

        // Pastikan password baru tidak sama dengan password lama
        if (Hash::check($request->new_password, $user->password)) {
            throw ValidationException::withMessages([
                'new_password' => 'Password baru tidak boleh sama dengan password lama.',
            ]);
        }

        // Update password
        $user->password = Hash::make($request->new_password);
        $user->save();

        return response()->json([
            'message' => 'Password berhasil diubah.',
        ]);
    }

    public function changeAvatar(Request $request)
    {
        $request->validate([
            'avatar' => ['required', 'image', 'mimes:jpeg,png,jpg,webp', 'max:2048'],
        ]);

        $user = Auth::user();

        // Hapus avatar lama jika ada
        if ($user->avatar && Storage::disk('public')->exists($user->avatar)) {
            Storage::disk('public')->delete($user->avatar);
        }

        // Simpan avatar baru
        $path = $request->file('avatar')->store('avatars', 'public');

        // Update user
        $user->avatar = $path;
        $user->save();

        return response()->json([
            'message' => 'Avatar berhasil diperbarui.',
            'avatar_url' => asset('storage/' . $path),
        ]);
    }

    public function deleteAccount(Request $request)
    {
        $user = Auth::user();

        // Validasi password sebelum hapus akun (opsional tapi disarankan)
        $request->validate([
            'password' => ['required'],
        ]);

        if (!Hash::check($request->password, $user->password)) {
            return response()->json([
                'message' => 'Password tidak sesuai.'
            ], 403);
        }

        // Hapus avatar dari storage
        if ($user->avatar && Storage::disk('public')->exists($user->avatar)) {
            Storage::disk('public')->delete($user->avatar);
        }

        // Hapus akun user
        $user->delete();

        return response()->json([
            'message' => 'Akun Anda telah dihapus secara permanen.'
        ]);
    }
}
