<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\User;
use Google\Client as GoogleClient;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use PHPOpenSourceSaver\JWTAuth\Facades\JWTAuth as FacadesJWTAuth;
use Tymon\JWTAuth\Facades\JWTAuth;

class GoogleJwtController extends Controller
{
    /**
     * Terima Google ID Token dari Flutter / Postman,
     * verifikasi, upsert user, lalu terbitkan JWT access token.
     */
    public function loginWithIdToken(Request $request)
    {
        $v = Validator::make($request->all(), [
            'id_token' => 'required|string',
        ]);
        if ($v->fails()) {
            return response()->json(['error' => $v->errors()->first()], 422);
        }

        // Verifikasi ID Token
        $client = new GoogleClient(['client_id' => env('GOOGLE_CLIENT_ID')]);
        $payload = $client->verifyIdToken($request->id_token);

        if (!$payload) {
            return response()->json(['error' => 'Invalid ID token'], 401);
        }
        if (!in_array($payload['iss'] ?? '', ['accounts.google.com', 'https://accounts.google.com'])) {
            return response()->json(['error' => 'Invalid issuer'], 401);
        }
        if (empty($payload['email_verified'])) {
            return response()->json(['error' => 'Unverified Google email'], 400);
        }

        $user = User::updateOrCreate(
            ['google_id' => $payload['sub']],
            [
                'name' => $payload['name'] ?? null,
                'email'  => $payload['email']   ?? null,
                'avatar' => $payload['picture'] ?? null,
            ]
        );


        $isNew = $user->wasRecentlyCreated;
        // Terbitkan access token JWT (TTL sesuai config/jwt.php)
        $accessToken = FacadesJWTAuth::fromUser($user);

        return response()->json([
            'token' => $accessToken,
            'token_type'   => 'Bearer',
            'expires_in'   => auth('api')->factory()->getTTL() * 60, // detik
            'is_new'       => $isNew,
            'user'         => $user,
        ]);
    }

    /**
     * Refresh access token.
     * Catatan: tymon/jwt-auth memakai token saat ini (dalam masa refresh_ttl)
     * untuk meminta token baru.
     */
    public function refresh()
    {
        try {
            $newToken = auth('api')->refresh(); // butuh Authorization: Bearer <token>
            return response()->json([
                'token' => $newToken,
                'token_type'   => 'Bearer',
                'expires_in'   => auth('api')->factory()->getTTL() * 60,
            ]);
        } catch (\Throwable $e) {
            return response()->json(['error' => 'Refresh failed'], 401);
        }
    }

    /**
     * Logout: blacklist token saat ini (jika blacklist diaktifkan).
     */
    public function logout()
    {
        try {
            auth('api')->logout(true);
            return response()->json(['ok' => true]);
        } catch (\Throwable $e) {
            return response()->json(['error' => 'Logout failed'], 401);
        }
    }
}
