<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Models\Alumni;
use App\Models\Category;
use App\Models\Footer;
use App\Models\Menu;
use App\Models\Section;
use App\Models\Banner;
use App\Models\Mentor;
use Illuminate\Http\Request;
use Exception;

class LandingController extends Controller
{
    public function header()
    {
        try {
            $data = Menu::orderBy('order')->get();

            return response()->json([
                'success' => true,
                'message' => 'Header data retrieved successfully',
                'data'    => $data,
            ]);
        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve header data',
                'error'   => $e->getMessage(),
            ], 500);
        }
    }

    public function landing()
    {
        try {
            $sections = Section::all()->map(function ($section) {
                return [
                    'id'          => $section->id,
                    'title'       => $this->formatPath($section->title),
                    'description' => $section->description,
                    'content'     => $this->formatPath($section->content),
                    'url'         => $section->url,
                    'type'        => $section->type,
                ];
            });

            $banners = Banner::all()->map(function ($banner) {
                return [
                    'id'    => $banner->id,
                    'title' => $banner->title,
                    'description' => $banner->description,
                    'image' => $this->formatPath($banner->image),
                    'url1'   => $banner->url1,
                    'url2'   => $banner->url2,
                ];
            });

            return response()->json([
                'success' => true,
                'message' => 'Landing data retrieved successfully',
                'data'    => [
                    'sections' => $sections,
                    'banners'  => $banners,
                ],
            ]);
        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve landing data',
                'error'   => $e->getMessage(),
            ], 500);
        }
    }

    public function footer()
    {
        try {
            $data = Footer::all();

            return response()->json([
                'success' => true,
                'message' => 'Footer data retrieved successfully',
                'data'    => $data,
            ]);
        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve footer data',
                'error'   => $e->getMessage(),
            ], 500);
        }
    }

    private function formatPath($value)
    {
        if (!$value) {
            return null;
        }

        if (filter_var($value, FILTER_VALIDATE_URL)) {
            return $value;
        }

        return asset('storage/' . ltrim($value, '/'));
    }

    public function course()
    {
        try {
            $categories = Category::with(['topics.courses'])->get();

            $data = $categories->map(function ($category) {
                return [
                    'id'    => $category->id,
                    'name'  => $category->name,
                    'icon'  => $category->icon,
                    'topics' => $category->topics->map(function ($topic) {
                        return [
                            'id'   => $topic->id,
                            'name' => $topic->name,
                            'icon' => $topic->icon,
                            'courses' => $topic->courses->map(function ($course) {
                                return [
                                    'id'          => $course->id,
                                    'title'       => $course->title,
                                    'image'       => asset('storage/' . $course->image),
                                    'price'       => $course->price,
                                    'free'        => $course->free,
                                    'description' => $course->description,
                                    'rating'      => $course->rating,
                                    'total_review'=> $course->total_review,
                                    'prize'       => $course->prize,
                                    'deadline'    => $course->deadline,
                                    'url'         => $course->url,
                                ];
                            }),
                        ];
                    }),
                ];
            });

            return response()->json([
                'success' => true,
                'message' => 'Course data retrieved successfully',
                'data'    => $data,
            ]);
        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve course data',
                'error'   => $e->getMessage(),
            ], 500);
        }
    }

    public function mentor()
    {
        try {
            $mentors = Mentor::with(['courses.topic'])->get();

            $data = $mentors->map(function ($mentor) {
                return [
                    'id'            => $mentor->id,
                    'name'          => $mentor->name,
                    'image'         => asset('storage/' . $mentor->image),
                    'rating'        => $mentor->rating,
                    'total_review'  => $mentor->total_review,
                    'achievement'   => $mentor->achievement,
                    'certification' => $mentor->certification,
                    'deadline'      => $mentor->deadline,
                    'topics'        => $mentor->courses->map(function ($course) {
                        return [
                            'id'   => $course->topic->id ?? null,
                            'name' => $course->topic->name ?? null,
                        ];
                    })->unique('id')->values(),
                ];
            });

            return response()->json([
                'success' => true,
                'message' => 'Mentor data retrieved successfully',
                'data'    => $data
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve mentor data',
                'error'   => $e->getMessage()
            ], 500);
        }
    }

    public function alumni()
    {
        try {
            $alumnis = Alumni::all();

            $data = $alumnis->map(function ($alumni) {
                return [
                    'id'          => $alumni->id,
                    'name'        => $alumni->name,
                    'image1'       => asset('storage/' . $alumni->image1),
                    'image2'       => asset('storage/' . $alumni->image2),
                    'occupation'  => $alumni->occupation,
                    'testimonial' => $alumni->testimonial,
                ];
            });

            return response()->json([
                'success' => true,
                'message' => 'Alumni data retrieved successfully',
                'data'    => $data
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error: ' . $e->getMessage(),
                'data'    => []
            ], 500);
        }
    }
}